<?php

/*********************************************************************

 Main File

*********************************************************************/

import('libs/cores/version.php');
import('libs/cores/regexp.php');
import('libs/cores/info.php');
import('libs/cores/db.php');

session();

db_connect();

normalize();

routing();

if (DEBUG_LEVEL and empty($params)) {
	switch ($_REQUEST['mode']) {
		case 'info_php':
			info_php();
		case 'info_fw':
			info_fw();
		case 'db_admin':
			db_admin();
	}
}

model();

controller();

view();

exit;

/********* Functions ************************************************/

function import($file)
{
	global $params, $db, $view;

	$flag = false;

	foreach (array(MAIN_APPLICATION_PATH, MAIN_LIBRARY_PATH) as $dir) {
		$target = dirname($_SERVER['SCRIPT_FILENAME']) . '/' . $dir . $file;
	
		if (is_file($target)) {
			require $target;

			$flag = true;

			break;
		}
	}

	if ($flag == false) {
		error('import error.' . (DEBUG_LEVEL ? ' [' . $target . ']' : ''));
	}

	return;
}

function session()
{
	session_set_cookie_params(SESSION_LIFETIME, SESSION_PATH);
	session_cache_limiter(SESSION_CACHE);
	session_start();

	return;
}

function normalize()
{
	if (ini_get('magic_quotes_gpc')) {
		$_GET     = unescape($_GET);
		$_POST    = unescape($_POST);
		$_REQUEST = unescape($_REQUEST);
		$_SERVER  = unescape($_SERVER);
		$_COOKIE  = unescape($_COOKIE);
	}

	$_GET     = sanitize($_GET);
	$_POST    = sanitize($_POST);
	$_REQUEST = sanitize($_REQUEST);
	$_SERVER  = sanitize($_SERVER);
	$_COOKIE  = sanitize($_COOKIE);

	$_GET     = unify($_GET);
	$_POST    = unify($_POST);
	$_REQUEST = unify($_REQUEST);
	$_SERVER  = unify($_SERVER);
	$_COOKIE  = unify($_COOKIE);

	return;
}

function routing()
{
	global $params;

	$request_uri = explode('/', strtok($_SERVER['REQUEST_URI'], '?'));
	$script_name = explode('/', $_SERVER['SCRIPT_NAME']);

	for ($i = 0; $i < sizeof($script_name); $i++) {
		if ($request_uri[$i] == $script_name[$i]) {
			unset($request_uri[$i]);
		}
	}
	$params = array_values(array_map('urldecode', $request_uri));

	if (count($params)) {
		if ($regexp = regexp_match('(.+)\.(\w*)$', $params[count($params) - 1])) {
			$params[count($params) - 1] = $regexp[1];
			$params[count($params)]     = $regexp[2];
		}
	}

	$_REQUEST = array(
		'mode'  => isset($_POST['mode'])  ? $_POST['mode']  : (isset($_GET['mode'])  ? $_GET['mode']  : null),
		'work'  => isset($_POST['work'])  ? $_POST['work']  : (isset($_GET['work'])  ? $_GET['work']  : null),
		'type'  => isset($_POST['type'])  ? $_POST['type']  : (isset($_GET['type'])  ? $_GET['type']  : null),
		'token' => isset($_POST['token']) ? $_POST['token'] : (isset($_GET['token']) ? $_GET['token'] : null)
	);

	if (isset($params[0])) {
		$_REQUEST['mode'] = $params[0];
	}
	if (isset($params[1])) {
		$_REQUEST['work'] = $params[1];
	}

	if ($_REQUEST['mode'] == '' or !regexp_match('^[_a-zA-Z0-9\-]+$', $_REQUEST['mode'])) {
		$_REQUEST['mode'] = 'default';
	}
	if ($_REQUEST['work'] == '' or !regexp_match('^[_a-zA-Z0-9\-]+$', $_REQUEST['work'])) {
		$_REQUEST['work'] = 'default';
	}

	return;
}

function model()
{
	$dir = 'app/models/';
	$php = '';

	if ($dh = opendir(MAIN_APPLICATION_PATH . $dir)) {
		while (($entry = readdir($dh)) !== false) {
			if (!is_file(MAIN_APPLICATION_PATH  . $dir . $entry)) {
				continue;
			}

			if ($regexp = regexp_match('^([_a-zA-Z0-9\-]+)\.php$', $entry)) {
				$name = $regexp[1];
			} else {
				continue;
			}

			import($dir . $entry);

			if (!function_exists('select_' . $name)) {
				$php .= 'function select_' . $name . '($queries)';
				$php .= '{';
				$php .= '  $queries["from"] = "' . DATABASE_PREFIX . $name . '";';
				$php .= '  return db_select($queries);';
				$php .= '}';
			}
			if (!function_exists('insert_' . $name)) {
				$php .= 'function insert_' . $name . '($queries)';
				$php .= '{';
				$php .= '  $queries["insert_into"] = "' . DATABASE_PREFIX . $name . '";';
				$php .= '  return db_insert($queries);';
				$php .= '}';
			}
			if (!function_exists('update_' . $name)) {
				$php .= 'function update_' . $name . '($queries)';
				$php .= '{';
				$php .= '  $queries["update"] = "' . DATABASE_PREFIX . $name . '";';
				$php .= '  return db_update($queries);';
				$php .= '}';
			}
			if (!function_exists('delete_' . $name)) {
				$php .= 'function delete_' . $name . '($queries)';
				$php .= '{';
				$php .= '  $queries["delete_from"] = "' . DATABASE_PREFIX . $name . '";';
				$php .= '  return db_delete($queries);';
				$php .= '}';
			}
			if (!function_exists('normalize_' . $name)) {
				$php .= 'function normalize_' . $name . '($queries)';
				$php .= '{';
				$php .= '  return $queries;';
				$php .= '}';
			}
			if (!function_exists('validate_' . $name)) {
				$php .= 'function validate_' . $name . '($queries)';
				$php .= '{';
				$php .= '  return array();';
				$php .= '}';
			}
		}
		closedir($dh);
	} else {
		error('opendir error.' . (DEBUG_LEVEL ? ' [' . $dir . ']' : ''));
	}

	eval($php);

	return;
}

function controller()
{
	global $params, $db, $view;

	$directory = 'app/controllers/' . $_REQUEST['mode'] . '/';
	$file      = $_REQUEST['work'] . '.php';

	if (is_file(MAIN_APPLICATION_PATH . $directory . $file)) {
		import($directory . $file);
	} elseif (is_file(MAIN_APPLICATION_PATH  . $directory . 'default.php')) {
		import($directory . 'default.php');
	}

	return;
}

function view($target = null)
{
	global $params, $view;

	static $complete = false;

	if ($complete) {
		return;
	}

	$directory = 'app/views/' . $_REQUEST['mode'] . '/';
	$file      = $_REQUEST['work'] . '.php';

	if ($target and is_file(MAIN_APPLICATION_PATH . $target)) {
		import($target);
	} elseif (is_file(MAIN_APPLICATION_PATH . $directory . $file)) {
		import($directory . $file);
	} elseif (is_file(MAIN_APPLICATION_PATH . $directory . 'default.php')) {
		import($directory . 'default.php');
	} elseif ($_REQUEST['mode'] == 'default' and $_REQUEST['work'] == 'default') {
		about();
	} else {
		import('app/views/404.php');
	}

	$complete = true;

	return;
}

function unescape($data)
{
	if (is_array($data)) {
		return array_map('unescape', $data);
	}

	return stripslashes($data);
}

function sanitize($data)
{
	if (is_array($data)) {
		return array_map('sanitize', $data);
	}

	return str_replace("\0", '', $data);
}

function unify($data)
{
	if (is_array($data)) {
		return array_map('unify', $data);
	}

	$data = regexp_replace("\r?\n", "\r", $data);
	$data = regexp_replace("\r", "\n", $data);

	return $data;
}

function convert($data, $to_encoding = 'UTF-8', $from_encoding = 'UTF-8,EUCJP-WIN,SJIS-WIN')
{
	if (mb_convert_variables($to_encoding, $from_encoding, $data)) {
		return $data;
	} else {
		return array();
	}
}

function token($type, $name = 'default')
{
	if ($type == 'check') {
		if ($_REQUEST['token'] and isset($_SESSION['core']['token'][$name]) and $_REQUEST['token'] == $_SESSION['core']['token'][$name]['value']) {
			$flag = true;
		} else {
			$flag = false;
		}

		if (empty($_SESSION['core']['token'][$name]) or time() - $_SESSION['core']['token'][$name]['time'] > TOKEN_SPAN) {
			$_SESSION['core']['token'][$name] = array();
		}

		return $flag;
	} else {
		if (empty($_SESSION['core']['token'][$name]) or time() - $_SESSION['core']['token'][$name]['time'] > TOKEN_SPAN) {
			$token = md5(uniqid(rand(), true));

			$_SESSION['core']['token'][$name] = array(
				'value' => $token,
				'time'  => time()
			);
		} else {
			$token = $_SESSION['core']['token'][$name]['value'];
		}

		return $token;
	}
}

function redirect($url)
{
	if (!regexp_match('^https?\:\/\/', $url)) {
		$url = MAIN_FILE . $url;
	}

	header('Location: ' . $url);

	exit;
}

function localdate($format = null, $timestamp = null)
{
	static $time = 0;

	if ($time == 0) {
		$time = time() + MAIN_TIME;
	}
	if ($timestamp == null) {
		$timestamp = $time;
	} elseif (!is_numeric($timestamp)) {
		$timestamp = strtotime($timestamp);
	}

	if ($format) {
		return date($format, $timestamp);
	} else {
		return $timestamp;
	}
}

function t($data, $return = false)
{
	$data = htmlspecialchars($data, ENT_QUOTES, MAIN_INTERNAL_ENCODING);

	if ($return) {
		return $data;
	} else {
		echo $data;
	}
}

function h($data, $return = false)
{
	$data = htmlspecialchars($data, ENT_QUOTES, MAIN_INTERNAL_ENCODING);
	$data = nl2br($data);

	if ($return) {
		return $data;
	} else {
		echo $data;
	}
}

function logging($message)
{
	if ($fp = fopen(LOGGING_FILE, 'a')) {
		fwrite($fp, '[' . localdate('Y-m-d H:i:s') . '] ' . $message . "\n");
		fclose($fp);
	}

	return;
}

function about()
{
	echo "<!DOCTYPE html>\n";
	echo "<html lang=\"ja\">\n";
	echo "<head>\n";
	echo "<meta charset=\"" . t(MAIN_CHARSET, true) . "\" />\n";
	echo "<title>About Framework</title>\n";
	echo "</head>\n";
	echo "<body>\n";
	echo "<h1>About Framework</h1>\n";
	echo "<p>levis: PHP Framework</p>\n";

	if (DEBUG_LEVEL) {
		echo "<p>Version " . VERSION_NUMBER . ' (' . VERSION_UPDATE . ")</p>\n";

		echo "<h2>Menu</h2>\n";
		echo "<ul>\n";
		echo "<li><a href=\"" . t(MAIN_FILE, true) . "?mode=info_php\">phpinfo</a></li>\n";

		if (DATABASE_TYPE) {
			echo "<li><a href=\"" . t(MAIN_FILE, true) . "?mode=db_admin\">database</a></li>\n";
		}

		echo "</ul>\n";

		echo "<h2>Configuration</h2>\n";

		echo "<h3>Main</h3>\n";
		echo "<dl>\n";
		echo "<dt>main file</dt>\n";
		echo "<dd><code>" . MAIN_FILE . "</code></dd>\n";
		echo "<dt>path to libraries</dt>\n";
		echo "<dd><code>" . (MAIN_LIBRARY_PATH ? MAIN_LIBRARY_PATH : '-') . "</code></dd>\n";
		echo "<dt>path to application</dt>\n";
		echo "<dd><code>" . (MAIN_APPLICATION_PATH ? MAIN_APPLICATION_PATH : '-') . "</code></dd>\n";
		echo "<dt>internal encoding</dt>\n";
		echo "<dd><code>" . MAIN_INTERNAL_ENCODING . "</code></dd>\n";
		echo "<dt>charset</dt>\n";
		echo "<dd><code>" . MAIN_CHARSET . "</code></dd>\n";
		echo "</dl>\n";
		echo "<dl>\n";
		echo "<dt>path to models</dt>\n";
		echo "<dd><code>" . MAIN_APPLICATION_PATH . 'app/models/' . "</code></dd>\n";
		echo "<dt>path to views</dt>\n";
		echo "<dd><code>" . MAIN_APPLICATION_PATH . 'app/views/' . "</code></dd>\n";
		echo "<dt>path to controllers</dt>\n";
		echo "<dd><code>" . MAIN_APPLICATION_PATH . 'app/controllers/' . "</code></dd>\n";
		echo "<dt>path to core libraries</dt>\n";
		echo "<dd><code>" . MAIN_LIBRARY_PATH . 'libs/cores/' . "</code></dd>\n";
		echo "<dt>path to plugins</dt>\n";
		echo "<dd><code>" . MAIN_LIBRARY_PATH . 'libs/plugins/' . "</code></dd>\n";
		echo "</dl>\n";

		if (DATABASE_TYPE) {
			echo "<h3>Database</h3>\n";
			echo "<dl>\n";
			echo "<dt>type</dt>\n";
			echo "<dd><code>" . DATABASE_TYPE . "</code></dd>\n";
			echo "<dt>name</dt>\n";
			echo "<dd><code>" . DATABASE_NAME . "</code></dd>\n";
			echo "<dt>prefix</dt>\n";
			echo "<dd><code>" . (DATABASE_PREFIX ? DATABASE_PREFIX : '-') . "</code></dd>\n";
			echo "<dt>charset</dt>\n";
			echo "<dd><code>" . (DATABASE_CHARSET ? DATABASE_CHARSET : '-') . "</code></dd>\n";
			echo "<dt>charset input from</dt>\n";
			echo "<dd><code>" . (DATABASE_CHARSET_INPUT_FROM ? DATABASE_CHARSET_INPUT_FROM : '-') . "</code></dd>\n";
			echo "<dt>charset input to</dt>\n";
			echo "<dd><code>" . (DATABASE_CHARSET_INPUT_TO ? DATABASE_CHARSET_INPUT_TO : '-') . "</code></dd>\n";
			echo "<dt>charset output from</dt>\n";
			echo "<dd><code>" . (DATABASE_CHARSET_OUTPUT_FROM ? DATABASE_CHARSET_OUTPUT_FROM : '-') . "</code></dd>\n";
			echo "<dt>charset output to</dt>\n";
			echo "<dd><code>" . (DATABASE_CHARSET_OUTPUT_TO ? DATABASE_CHARSET_OUTPUT_TO : '-') . "</code></dd>\n";
			echo "</dl>\n";
		}

		echo "<h3>Session</h3>\n";
		echo "<dl>\n";
		echo "<dt>lifetime</dt>\n";
		echo "<dd><code>" . SESSION_LIFETIME . "</code></dd>\n";
		echo "<dt>path</dt>\n";
		echo "<dd><code>" . SESSION_PATH . "</code></dd>\n";
		echo "<dt>cache</dt>\n";
		echo "<dd><code>" . SESSION_CACHE . "</code></dd>\n";
		echo "</dl>\n";

		echo "<h3>Token</h3>\n";
		echo "<dl>\n";
		echo "<dt>span</dt>\n";
		echo "<dd><code>" . TOKEN_SPAN . "</code></dd>\n";
		echo "</dl>\n";

		echo "<h3>Regexp</h3>\n";
		echo "<dl>\n";
		echo "<dt>type</dt>\n";
		echo "<dd><code>" . REGEXP_TYPE . "</code></dd>\n";
		echo "</dl>\n";

		echo "<h3>Logging</h3>\n";
		echo "<dl>\n";
		echo "<dt>file</dt>\n";
		echo "<dd><code>" . LOGGING_FILE . "</code></dd>\n";
		echo "</dl>\n";

		echo "<h3>Debug</h3>\n";
		echo "<dl>\n";
		echo "<dt>level</dt>\n";
		echo "<dd><code>" . DEBUG_LEVEL . "</code></dd>\n";
		echo "<dt>log</dt>\n";
		echo "<dd><code>" . DEBUG_LOG . "</code></dd>\n";
		echo "</dl>\n";
	}

	echo "</body>\n";
	echo "</html>\n";

	exit;
}

function ok($type = null)
{
	if ($type == null) {
		$type = $_REQUEST['type'];
	}

	if ($type == 'json') {
		header('Content-Type: application/json; charset=' . MAIN_CHARSET);

		echo json_encode(array(
			'status' => 'OK',
		));
	} elseif ($type == 'xml') {
		header('Content-Type: text/xml; charset=' . MAIN_CHARSET);

		echo "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n";
		echo "<response>\n";
		echo "<status>OK</status>\n";
		echo "</response>\n";
	} else {
		echo "<!DOCTYPE html>\n";
		echo "<html lang=\"ja\">\n";
		echo "<head>\n";
		echo "<meta charset=\"" . t(MAIN_CHARSET, true) . "\" />\n";
		echo "<title>OK</title>\n";
		echo "</head>\n";
		echo "<body>\n";
		echo "<h1>OK</h1>\n";
		echo "</body>\n";
		echo "</html>\n";
	}

	exit;
}

function warning($messages, $type = null)
{
	if (!is_array($messages)) {
		$messages = array($messages);
	}
	if ($type == null) {
		$type = $_REQUEST['type'];
	}

	if ($type == 'json') {
		header('Content-Type: application/json; charset=' . MAIN_CHARSET);

		echo json_encode(array(
			'status'   => 'WARNING',
			'messages' => $messages,
		));
	} elseif ($type == 'xml') {
		header('Content-Type: text/xml; charset=' . MAIN_CHARSET);

		echo "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n";
		echo "<response>\n";
		echo "<status>WARNING</status>\n";
		echo "<messages>\n";

		foreach ($messages as $message) {
			echo "<message>" . h($message, true) . "</message>\n";
		}

		echo "</messages>\n";
		echo "</response>\n";
	} else {
		echo "<!DOCTYPE html>\n";
		echo "<html lang=\"ja\">\n";
		echo "<head>\n";
		echo "<meta charset=\"" . t(MAIN_CHARSET, true) . "\" />\n";
		echo "<title>WARNING</title>\n";
		echo "</head>\n";
		echo "<body>\n";
		echo "<h1>WARNING</h1>\n";
		echo "<ul>\n";

		foreach ($messages as $message) {
			echo "<li>" . h($message, true) . "</li>\n";
		}

		echo "</ul>\n";
		echo "</body>\n";
		echo "</html>\n";
	}

	exit;
}

function error($message, $type = null)
{
	if (DEBUG_LOG) {
		logging($message);
	}
	if ($type == null) {
		$type = $_REQUEST['type'];
	}

	if ($type == 'json') {
		header('Content-Type: application/json; charset=' . MAIN_CHARSET);

		echo json_encode(array(
			'status'  => 'ERROR',
			'message' => $message,
		));
	} elseif ($type == 'xml') {
		header('Content-Type: text/xml; charset=' . MAIN_CHARSET);

		echo "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n";
		echo "<response>\n";
		echo "<status>ERROR</status>\n";
		echo "<message>" . h($message, true) . "</message>\n";
		echo "</response>\n";
	} else {
		echo "<!DOCTYPE html>\n";
		echo "<html lang=\"ja\">\n";
		echo "<head>\n";
		echo "<meta charset=\"" . t(MAIN_CHARSET, true) . "\" />\n";
		echo "<title>ERROR</title>\n";
		echo "</head>\n";
		echo "<body>\n";
		echo "<h1>ERROR</h1>\n";
		echo "<p>" . h($message, true) . "</p>\n";
		echo "</body>\n";
		echo "</html>\n";
	}

	exit;
}

?>
