<?php

/*********************************************************************

 Functions for DB

*********************************************************************/

function db_connect()
{
	global $db;

	if (DATABASE_TYPE == '') {
		return;
	}

	if (DATABASE_TYPE == 'pdo_mysql' or DATABASE_TYPE == 'pdo_pgsql' or DATABASE_TYPE == 'pdo_sqlite' or DATABASE_TYPE == 'pdo_sqlite2') {
		import('libs/cores/db_pdo.php');
	} elseif (DATABASE_TYPE == 'mysql') {
		import('libs/cores/db_mysql.php');
	} elseif (DATABASE_TYPE == 'pgsql') {
		import('libs/cores/db_pgsql.php');
	} elseif (DATABASE_TYPE == 'sqlite') {
		import('libs/cores/db_sqlite.php');
	}

	db_driver_connect();

	if ((DATABASE_TYPE == 'pdo_mysql' or DATABASE_TYPE == 'mysql' or DATABASE_TYPE == 'pdo_pgsql' or DATABASE_TYPE == 'pgsql') and DATABASE_CHARSET) {
		$resource = db_query('SET NAMES \'' . DATABASE_CHARSET . '\'');
		if (!$resource) {
			error('database set names error.' . (DEBUG_LEVEL ? ' [' . DATABASE_CHARSET . ']' : ''));
		}
	}

	return;
}

function db_query($query, $error = true)
{
	global $db;

	if (DATABASE_TYPE == '') {
		return false;
	}

	if (DATABASE_CHARSET_INPUT_TO != DATABASE_CHARSET_INPUT_FROM) {
		$query = convert($query, DATABASE_CHARSET_INPUT_TO, DATABASE_CHARSET_INPUT_FROM);
	}

	if (DEBUG_LEVEL == 2) {
		echo '<pre><code>' . $query . '</code></pre>';
	}

	$resource = db_driver_query($query);

	if (!$resource and $error) {
		error('database query error.' . (DEBUG_LEVEL ? ' [' . db_error() . ']' : ''));
	}

	return $resource;
}

function db_result($resource)
{
	global $db;

	if (DATABASE_TYPE == '') {
		return array();
	}

	$results = db_driver_result($resource);

	if (DATABASE_CHARSET_OUTPUT_TO != DATABASE_CHARSET_OUTPUT_FROM) {
		$results = convert($results, DATABASE_CHARSET_OUTPUT_TO, DATABASE_CHARSET_OUTPUT_FROM);
	}

	return $results;
}

function db_count($resource)
{
	global $db;

	if (DATABASE_TYPE == '') {
		return -1;
	}

	return db_driver_count($resource);
}

function db_affected_count($resource)
{
	global $db;

	if (DATABASE_TYPE == '') {
		return -1;
	}

	return db_driver_affected_count($resource);
}

function db_escape($data)
{
	global $db;

	if (DATABASE_TYPE == '') {
		return $data;
	}

	return db_driver_escape($data);
}

function db_error()
{
	global $db;

	if (DATABASE_TYPE == '') {
		return '';
	}

	return db_driver_error();
}

function db_select($queries)
{
	global $db;

	if (DATABASE_TYPE == '') {
		return array();
	}

	if (isset($queries['select']) and $queries['select'] != '') {
		$queries['select'] = 'SELECT ' . $queries['select'] . ' ';
	} else {
		$queries['select'] = 'SELECT * ';
	}
	if (isset($queries['from']) and $queries['from'] != '') {
		$queries['from'] = 'FROM ' . $queries['from'] . ' ';
	} else {
		return array();
	}

	if (isset($queries['where']) and $queries['where'] != '') {
		$queries['where'] = 'WHERE ' . $queries['where'] . ' ';
	} else {
		$queries['where'] = '';
	}
	if (isset($queries['group_by']) and $queries['group_by'] != '') {
		$queries['group_by'] = 'GROUP BY ' . $queries['group_by'] . ' ';
	} else {
		$queries['group_by'] = '';
	}
	if (isset($queries['having']) and $queries['having'] != '') {
		$queries['having'] = 'HAVING ' . $queries['having'] . ' ';
	} else {
		$queries['having'] = '';
	}
	if (isset($queries['order_by']) and $queries['order_by'] != '') {
		$queries['order_by'] = 'ORDER BY ' . $queries['order_by'] . ' ';
	} else {
		$queries['order_by'] = '';
	}
	if (isset($queries['offset']) and $queries['offset'] != '') {
		$queries['offset'] = 'OFFSET ' . $queries['offset'] . ' ';
	} else {
		$queries['offset'] = '';
	}
	if (isset($queries['limit']) and $queries['limit'] != '') {
		$queries['limit'] = 'LIMIT ' . $queries['limit'] . ' ';
	} else {
		$queries['limit'] = '';
	}

	$resource = db_query($queries['select'] . $queries['from'] . $queries['where'] . $queries['group_by'] . $queries['having'] . $queries['order_by'] . $queries['offset'] . $queries['limit']);

	return db_result($resource);
}

function db_insert($queries)
{
	global $db;

	if (DATABASE_TYPE == '') {
		return false;
	}

	if (isset($queries['insert_into']) and $queries['insert_into'] != '') {
		$queries['insert_into'] = 'INSERT INTO ' . $queries['insert_into'] . ' ';
	} else {
		return false;
	}

	if (isset($queries['values']) and $queries['values'] != '') {
		$keys   = array();
		$values = array();

		foreach ($queries['values'] as $key => $value) {
			$keys[] = $key;

			if ($value === null or $value === '' or $value === '\'\'') {
				$values[] = 'NULL';
			} else {
				$values[] = $value;
			}
		}

		$queries['values'] = '(' . implode(', ', $keys) . ') VALUES(' . implode(', ', $values) . ') ';
	} else {
		return false;
	}

	return db_query($queries['insert_into'] . $queries['values']);
}

function db_update($queries)
{
	global $db;

	if (DATABASE_TYPE == '') {
		return false;
	}

	if (isset($queries['update']) and $queries['update'] != '') {
		$queries['update'] = 'UPDATE ' . $queries['update'] . ' ';
	} else {
		return false;
	}

	if (isset($queries['sets']) and $queries['sets'] != '') {
		$sets = array();

		foreach ($queries['sets'] as $key => $value) {
			if ($value === null or $value === '' or $value === '\'\'') {
				$sets[] = $key . ' = NULL';
			} else {
				$sets[] = $key . ' = ' . $value;
			}
		}

		$queries['sets'] = 'SET ' . implode(', ', $sets) . ' ';
	} else {
		return false;
	}

	if (isset($queries['where']) and $queries['where'] != '') {
		$queries['where'] = 'WHERE ' . $queries['where'] . ' ';
	} else {
		$queries['where'] = '';
	}
	if (isset($queries['offset']) and $queries['offset'] != '') {
		$queries['offset'] = 'OFFSET ' . $queries['offset'] . ' ';
	} else {
		$queries['offset'] = '';
	}
	if (isset($queries['limit']) and $queries['limit'] != '') {
		$queries['limit'] = 'LIMIT ' . $queries['limit'] . ' ';
	} else {
		$queries['limit'] = '';
	}

	return db_query($queries['update'] . $queries['sets'] . $queries['where'] . $queries['offset'] . $queries['limit']);
}

function db_delete($queries)
{
	global $db;

	if (DATABASE_TYPE == '') {
		return false;
	}

	if (isset($queries['delete_from']) and $queries['delete_from'] != '') {
		$queries['delete_from'] = 'DELETE FROM ' . $queries['delete_from'] . ' ';
	} else {
		return false;
	}

	if (isset($queries['where']) and $queries['where'] != '') {
		$queries['where'] = 'WHERE ' . $queries['where'] . ' ';
	} else {
		$queries['where'] = '';
	}
	if (isset($queries['offset']) and $queries['offset'] != '') {
		$queries['offset'] = 'OFFSET ' . $queries['offset'] . ' ';
	} else {
		$queries['offset'] = '';
	}
	if (isset($queries['limit']) and $queries['limit'] != '') {
		$queries['limit'] = 'LIMIT ' . $queries['limit'] . ' ';
	} else {
		$queries['limit'] = '';
	}

	return db_query($queries['delete_from'] . $queries['where'] . $queries['offset'] . $queries['limit']);
}

function db_last_insert_id($resource)
{
	global $db;

	if (DATABASE_TYPE == '') {
		return -1;
	}

	return db_driver_last_insert_id($resource);
}

function db_transaction()
{
	global $db;

	if (DATABASE_TYPE == '') {
		return false;
	}

	return db_driver_transaction();
}

function db_commit()
{
	global $db;

	if (DATABASE_TYPE == '') {
		return false;
	}

	return db_driver_commit();
}

function db_rollback()
{
	global $db;

	if (DATABASE_TYPE == '') {
		return false;
	}

	return db_driver_rollback();
}

function db_admin()
{
	global $db;

	if (DATABASE_TYPE == '') {
		return false;
	}

	if ($_REQUEST['work'] == 'import') {
		db_admin_import();
	} elseif ($_REQUEST['work'] == 'export') {
		db_admin_export();
	} else {
		db_admin_sql();
	}

	exit;
}

function db_admin_import()
{
	global $db;

	if ($_SERVER['REQUEST_METHOD'] == 'POST') {
		if ($_POST['means'] == 'upload') {
			if (is_uploaded_file($_FILES['target']['tmp_name'])) {
				$target = $_FILES['target']['tmp_name'];
			} else {
				error('file not found.');
			}
		} else {
			$target = DATABASE_NAME . '.sql';

			if (!is_file($target)) {
				error('file not found.');
			}
		}

		if ($fp = fopen($target, 'r')) {
			$sql  = '';
			$i    = 0;
			$flag = true;

			db_transaction();

			while ($line = fgets($fp)) {
				$line = str_replace("\r\n", "\n", $line);
				$line = str_replace("\r", "\n", $line);

				if ((substr_count($line, '\'') - substr_count($line, '\\\'')) % 2 != 0) {
					$flag = !$flag;
				}

				$sql .= $line;

				if (preg_match('/;$/', trim($line)) and $flag) {
					$resource = db_query($sql, false);

					if (!$resource) {
						db_rollback();

						error('database query error.' . (DEBUG_LEVEL ? ' [' . db_error() . ']' : ''));
					}

					$sql = '';
					$i++;
				}
			}
			fclose($fp);

			db_commit();

			$message = $i . ' sql executed.';
		} else {
			error('file can\'t read.');
		}
	}

	echo "<!DOCTYPE html>\n";
	echo "<html lang=\"ja\">\n";
	echo "<head>\n";
	echo "<meta charset=\"" . t(MAIN_CHARSET, true) . "\" />\n";
	echo "<title>DB</title>\n";
	echo "</head>\n";
	echo "<body>\n";
	echo "<h1><a href=\"" . t(MAIN_FILE, true) . "?mode=db_admin\">DB</a></h1>\n";

	echo "<h2>Menu</h2>\n";
	echo "<ul>\n";
	echo "<li><a href=\"" . t(MAIN_FILE, true) . "?mode=db_admin&amp;work=sql\">SQL</a></li>\n";
	echo "<li>Import</li>\n";
	echo "<li><a href=\"" . t(MAIN_FILE, true) . "?mode=db_admin&amp;work=export\">Export</a></li>\n";
	echo "</ul>\n";

	echo "<h2>Import</h2>\n";

	if (isset($message)) {
		echo "<ul>\n";
		echo "<li>" . $message . "</li>\n";
		echo "</ul>\n";
	} else {
		echo "<ul>\n";
		echo "<li>Import from SQL file.</li>\n";
		echo "</ul>\n";
	}

	echo "<form action=\"" . t(MAIN_FILE, true) . "?mode=db_admin&amp;work=import\" method=\"post\" enctype=\"multipart/form-data\">\n";
	echo "<fieldset>\n";
	echo "<legend>import</legend>\n";
	echo "<dl>\n";
	echo "<dt><label><input type=\"radio\" name=\"means\" value=\"upload\" checked=\"checked\" /> upload</label></dt>\n";
	echo "<dd><input type=\"file\" name=\"target\" size=\"30\" /></dd>\n";
	echo "<dt><label><input type=\"radio\" name=\"means\" value=\"file\" /> read</label></dt>\n";
	echo "<dd><code>" . DATABASE_NAME . ".sql</code></dd>\n";
	echo "</dl>\n";
	echo "<p><input type=\"submit\" value=\"import\" /></p>\n";
	echo "</fieldset>\n";
	echo "</form>\n";

	echo "</body>\n";
	echo "</html>\n";

	return;
}

function db_admin_export()
{
	global $db;

	$resource = db_query(db_sql('table_list'));

	$results = db_result($resource);

	$tables = array();
	foreach ($results as $result) {
		$tables[] = array_pop($result);
	}

	if ($_SERVER['REQUEST_METHOD'] == 'POST') {
		$text  = '-- Database: ' . DATABASE_NAME . ' (' . DATABASE_TYPE . ")\n";
		$text .= '-- Datetime: ' . localdate('Y-m-d H:i:s') . "\n";
		$text .= '-- Host: ' . gethostbyaddr($_SERVER['REMOTE_ADDR']) . "\n";
		$text .= "\n";

		foreach ($tables as $table) {
			if (empty($_POST['table']) or $_POST['table'] == $table) {
				$resource = db_query(db_sql('table_create', $table));

				$results = db_result($resource);

				if (DATABASE_TYPE == 'pdo_mysql' or DATABASE_TYPE == 'mysql') {
					$text .= "DROP TABLE IF EXISTS " . $table . ";\n";
					$text .= $results[0]['Create Table'] . ";\n";
					$text .= "\n";
				} elseif (DATABASE_TYPE == 'pdo_pgsql' or DATABASE_TYPE == 'pgsql') {
					$text .= "DROP TABLE IF EXISTS " . $table . ";\n";
					$text .= $results[0]['case'] . ";\n";
					$text .= "\n";
				} elseif (DATABASE_TYPE == 'pdo_sqlite' or DATABASE_TYPE == 'pdo_sqlite2' or DATABASE_TYPE == 'sqlite') {
					$text .= "DROP TABLE IF EXISTS " . $table . ";\n";
					$text .= $results[0]['sql'] . ";\n";
					$text .= "\n";
				}

				$resource = db_query('SELECT * FROM ' . $table . ';');

				$results = db_result($resource);

				foreach ($results as $result) {
					$inserts = array();
					foreach ($result as $data) {
						if ($data === null) {
							$inserts[] = 'NULL';
						} else {
							$inserts[] = db_escape($data);
						}
					}
					$text .= "INSERT INTO " . $table . " VALUES(" . implode(',', $inserts) . ");\n";
				}

				$text .= "\n";
			}
		}

		if ($_POST['means'] == 'download') {
			header('Content-Type: text/plain');
			header('Content-Disposition: attachment; filename="' . DATABASE_NAME . '.sql"');
			echo $text;
			exit;
		} else {
			if (file_put_contents(DATABASE_NAME . '.sql', $text) === false) {
				error('write error.' . (DEBUG_LEVEL ? ' [' . DATABASE_NAME . '.sql' . ']' : ''));
			}

			$message = 'exported.';
		}
	}

	echo "<!DOCTYPE html>\n";
	echo "<html lang=\"ja\">\n";
	echo "<head>\n";
	echo "<meta charset=\"" . t(MAIN_CHARSET, true) . "\" />\n";
	echo "<title>DB</title>\n";
	echo "</head>\n";
	echo "<body>\n";
	echo "<h1><a href=\"" . t(MAIN_FILE, true) . "?mode=db_admin\">DB</a></h1>\n";

	echo "<h2>Menu</h2>\n";
	echo "<ul>\n";
	echo "<li><a href=\"" . t(MAIN_FILE, true) . "?mode=db_admin&amp;work=sql\">SQL</a></li>\n";
	echo "<li><a href=\"" . t(MAIN_FILE, true) . "?mode=db_admin&amp;work=import\">Import</a></li>\n";
	echo "<li>Export</li>\n";
	echo "</ul>\n";

	echo "<h2>Export</h2>\n";

	if (isset($message)) {
		echo "<ul>\n";
		echo "<li>" . $message . "</li>\n";
		echo "</ul>\n";
	} else {
		echo "<ul>\n";
		echo "<li>Export to SQL file.</li>\n";
		echo "</ul>\n";
	}

	echo "<form action=\"" . t(MAIN_FILE, true) . "?mode=db_admin&amp;work=export\" method=\"post\" enctype=\"multipart/form-data\">\n";
	echo "<fieldset>\n";
	echo "<legend>export</legend>\n";

	echo "<dl>\n";
	echo "<dt>table</dt>\n";
	echo "<dd>\n";
	echo "<select name=\"table\">\n";
	echo "<option value=\"\">(all)</option>\n";

	foreach ($tables as $table) {
		echo "<option value=\"" . $table . "\">" . $table . "</option>\n";
	}

	echo "</select>\n";
	echo "</dd>\n";
	echo "<dt>means</dt>\n";
	echo "<dd>\n";
	echo "<label><input type=\"radio\" name=\"means\" value=\"download\" checked=\"checked\" /> download</label><br />\n";
	echo "<label><input type=\"radio\" name=\"means\" value=\"write\" /> write to <code>" . DATABASE_NAME . ".sql</code></label>\n";
	echo "</dd>\n";
	echo "</dl>\n";
	echo "<p><input type=\"submit\" value=\"export\" /></p>\n";
	echo "</fieldset>\n";
	echo "</form>\n";

	echo "</body>\n";
	echo "</html>\n";

	return;
}

function db_admin_sql()
{
	global $db;

	if ($_SERVER['REQUEST_METHOD'] == 'POST') {
		$sql = $_POST['sql'];
	} else {
		$sql = db_sql('table_list');
	}

	list($micro, $second) = explode(' ', microtime());
	$time_start = $micro + $second;

	$resource = db_query($sql);

	list($micro, $second) = explode(' ', microtime());
	$time_end = $micro + $second;

	$view['time'] = ceil(($time_end - $time_start) * 10000) / 10000;

	$view['sql'] = $sql;

	if ($sql == db_sql('table_list')) {
		$head = '';
		$body = '';

		$results = db_result($resource);

		$head .= '<tr>';
		$head .= '<th>name</th>';
		$head .= '<th>create</th>';
		$head .= '<th>columns</th>';
		$head .= '<th>drop</th>';
		$head .= '<th>insert</th>';
		$head .= '<th>delete</th>';
		$head .= '<th>select</th>';
		$head .= '</tr>';

		foreach ($results as $result) {
			$table = array_shift($result);

			if (DATABASE_TYPE == 'pdo_mysql' or DATABASE_TYPE == 'mysql') {
				$create     = 'SHOW CREATE TABLE';
				$define     = 'SHOW COLUMNS';
			} elseif (DATABASE_TYPE == 'pdo_pgsql' or DATABASE_TYPE == 'pgsql') {
				$create     = 'create';
				$define     = 'columns';
			} elseif (DATABASE_TYPE == 'pdo_sqlite' or DATABASE_TYPE == 'pdo_sqlite2' or DATABASE_TYPE == 'sqlite') {
				$create     = 'SELECT sql';
				$define     = 'PRAGMA TABLE_INFO';
			}
			$create_sql = db_sql('table_create', $table);
			$define_sql = db_sql('table_define', $table);

			$create_sql = preg_replace('/"/', '&quot;', $create_sql);
			$define_sql = preg_replace('/"/', '&quot;', $define_sql);

			$define_resource = db_query($define_sql);
			$define_results  = db_result($define_resource);

			$insert_keys   = array();
			$insert_values = array();

			foreach ($define_results as $define_result) {
				if (DATABASE_TYPE == 'pdo_mysql' or DATABASE_TYPE == 'mysql') {
					$insert_keys[]   = $define_result['Field'];
					$insert_values[] = $define_result['Null'] == 'YES' ? 'NULL' : '\\\'\\\'';
				} elseif (DATABASE_TYPE == 'pdo_pgsql' or DATABASE_TYPE == 'pgsql') {
					$insert_keys[]   = $define_result['column_name'];
					$insert_values[] = $define_result['is_nullable'] == 'YES' ? 'NULL' : '\\\'\\\'';
				} elseif (DATABASE_TYPE == 'pdo_sqlite' or DATABASE_TYPE == 'pdo_sqlite2' or DATABASE_TYPE == 'sqlite') {
					$insert_keys[]   = $define_result['name'];
					$insert_values[] = $define_result['notnull'] == 0 ? 'NULL' : '\\\'\\\'';
				}
			}

			$body .= '<tr>';
			$body .= '<td><span style="font-family:monospace;">' . $table . '</span></td>';
			$body .= '<td><a href="javascript:insertSQL(\'' . str_replace('\'', '\\\'', $create_sql) . '\');">' . $create . '</a></td>';
			$body .= '<td><a href="javascript:insertSQL(\'' . str_replace('\'', '\\\'', $define_sql) . '\');">' . $define . '</a></td>';
			$body .= '<td><a href="javascript:insertSQL(\'DROP TABLE ' . $table . ';\');">DROP TABLE</a></td>';
			$body .= '<td><a href="javascript:insertSQL(\'INSERT INTO ' . $table . '(' . implode(',', $insert_keys) . ') VALUES(' . implode(',', $insert_values) . ');\');">INSERT</a></td>';
			$body .= '<td><a href="javascript:insertSQL(\'DELETE FROM ' . $table . ';\');">DELETE</a></td>';
			$body .= '<td><a href="javascript:insertSQL(\'SELECT * FROM ' . $table . ' LIMIT 100;\');">SELECT</a></td>';
			$body .= '</tr>';
		}

		$view['result'] = '<table border="1">' . $head . $body . '</table>';
		$view['count']  = db_count($resource);
	} elseif (regexp_match('^(SELECT|SHOW|EXPLAIN|DESC|PRAGMA)', $sql)) {
		$head = '';
		$body = '';
		$flag = false;

		if ($regexp = regexp_match('^SELECT \* FROM ([_a-zA-Z0-9\-]+)', $sql)) {
			$table = $regexp[1];
		} else {
			$table = null;
		}

		$results = db_result($resource);

		foreach ($results as $result) {
			$first_key   = null;
			$first_value = null;

			$body .= '<tr>';

			foreach ($result as $key => $value) {
				if ($first_key === null) {
					$first_key   = $key;
					$first_value = $value;
				}

				if (is_string($key)) {
					if ($value === null) {
						$value_sql  = 'NULL';
						$value_html = '<em>NULL</em>';
					} else {
						$value_sql = str_replace('\\', '\\\\\\', $value);
						$value_sql = str_replace("\n", '\n', $value_sql);
						$value_sql = str_replace('"', '&quot;', $value_sql);

						if (DATABASE_TYPE == 'pdo_mysql' or DATABASE_TYPE == 'mysql') {
							$value_sql = str_replace('\'', '\\\\\\\'', $value_sql);
						} else {
							$value_sql = str_replace('\'', '\\\'\\\'', $value_sql);
						}

						$value_sql  = '\\\'' . $value_sql . '\\\'';
						$value_html = h($value, true);
					}

					if ($table === null) {
						$value = $value_html;
					} else {
						$value = '<a href="javascript:insertSQL(\'UPDATE ' . $table . ' SET ' . $key . ' = ' . $value_sql . ' WHERE ' . $first_key . ' = \\\'' . $first_value . '\\\';\');">' . $value_html . '</a>';
					}

					$body .= '<td><span style="font-family:monospace;">' . $value . '</span></td>';

					if ($flag == false) {
						$head .= '<th>' . h($key, true) . '</th>';
					}
				}
			}

			$body .= '</tr>';

			$flag = true;
		}

		$view['result'] = '<table border="1"><tr>' . $head . '</tr>' . $body . '</table>';
		$view['count']  = db_count($resource);
	} else {
		$view['result'] = '<p>OK</p>';
		$view['count']  = db_affected_count($resource);
	}

	echo "<!DOCTYPE html>\n";
	echo "<html lang=\"ja\">\n";
	echo "<head>\n";
	echo "<meta charset=\"" . t(MAIN_CHARSET, true) . "\" />\n";
	echo "<title>DB</title>\n";
	echo "<script>\n";
	echo "function insertSQL(sql)\n";
	echo "{\n";
	echo "\t\tdocument.getElementById('exec_form').sql.value = sql;\n";
	echo "}";
	echo "</script>\n";
	echo "</head>\n";
	echo "<body>\n";
	echo "<h1><a href=\"" . t(MAIN_FILE, true) . "?mode=db_admin\">DB</a></h1>\n";

	echo "<h2>Menu</h2>\n";
	echo "<ul>\n";
	echo "<li>SQL</li>\n";
	echo "<li><a href=\"" . t(MAIN_FILE, true) . "?mode=db_admin&amp;work=import\">Import</a></li>\n";
	echo "<li><a href=\"" . t(MAIN_FILE, true) . "?mode=db_admin&amp;work=export\">Export</a></li>\n";
	echo "</ul>\n";

	echo "<h2>SQL</h2>\n";
	echo "<form action=\"" . t(MAIN_FILE, true) . "?mode=db_admin\" method=\"post\" id=\"exec_form\">\n";
	echo "<fieldset>\n";
	echo "<legend>execute</legend>\n";
	echo "<dl>\n";
	echo "<dt>SQL</dt>\n";
	echo "<dd><textarea name=\"sql\" cols=\"50\" rows=\"5\">" . t($view['sql'], true) . "</textarea></dd>\n";
	echo "</dl>\n";
	echo "<p><input type=\"submit\" value=\"execute\" /></p>\n";
	echo "</fieldset>\n";
	echo "</form>\n";

	if ($view['result']) {
		echo "<h2>Result</h2>\n";
		echo $view['result'];
	}

	echo "<pre><code>rows = " . $view['count'] . " rows.\n";
	echo "time = " . $view['time'] . " sec.</code></pre>\n";

	echo "</body>\n";
	echo "</html>\n";

	return;
}

function db_sql($type, $table = null)
{
	global $db;

	if (DATABASE_TYPE == 'pdo_mysql' or DATABASE_TYPE == 'mysql') {
		if ($type == 'table_list') {
			$sql = '
				SHOW TABLES;
			';
		} elseif ($type == 'table_create') {
			$sql = '
				SHOW CREATE TABLE ' . $table . ';
			';
		} elseif ($type == 'table_define') {
			$sql = '
				SHOW COLUMNS FROM ' . $table . ';
			';
		}
	} elseif (DATABASE_TYPE == 'pdo_pgsql' or DATABASE_TYPE == 'pgsql') {
		if ($type == 'table_list') {
			$sql = '
				SELECT
					pg_class.relname AS relname
				FROM
					pg_class INNER JOIN pg_namespace ON pg_class.relnamespace = pg_namespace.oid
				WHERE
					pg_class.relkind = \'r\' AND pg_namespace.nspname = \'public\';
			';
		} elseif ($type == 'table_create') {
			$sql = '
				SELECT
					CASE
						WHEN tb.relkind = \'r\' THEN(
							SELECT \'CREATE TABLE "' . $table . '"(\' || chr(10) || array_to_string(
								ARRAY(
									SELECT \' "\' || "Column" || \'" \'|| "Type" || "Modifiers" || "Index" FROM(
										/* Column */
										SELECT
											at.attnum, ns.nspname AS schema, tb.relname AS table, at.attname AS "Column",
											/* Type */
											CASE
												WHEN at.attinhcount <> 0 OR at.attisdropped THEN null
												ELSE
													CASE
														WHEN tp.typname = \'int2\'   THEN \'SMALLINT\'
														WHEN tp.typname = \'int4\'   THEN \'INTEGER\'
														WHEN tp.typname = \'int8\'   THEN \'BIGINT\'
														WHEN tp.typname = \'float4\' THEN \'REAL\'
														WHEN tp.typname = \'float8\' THEN \'DOUBLE PRECISION\'
														WHEN tp.typname = \'bpchar\' THEN \'CHAR\'
														ELSE UPPER(tp.typname)
													END ||
													CASE
														WHEN at.attlen >= 0             THEN \'\'
														WHEN at.atttypmod < 4           THEN \'\'
														WHEN tp.typname <> \'numeric\'  THEN \'(\' || at.atttypmod - 4 || \')\'
														WHEN (at.atttypmod & 65535) = 4 THEN \'(\' || (at.atttypmod >> 16) || \')\'
														ELSE \'(\' || (at.atttypmod >> 16) || \',\' || (at.atttypmod & 65535) - 4 || \')\'
													END
											END AS "Type",
											/* Modifiers */
											CASE
												WHEN at.attnotnull THEN \' NOT NULL\'
												ELSE \'\'
											END ||
											CASE
												WHEN ad.adbin IS NULL THEN \'\'
												ELSE \' DEFAULT \' || UPPER(pg_get_expr(ad.adbin, tb.oid))
											END AS "Modifiers",
											/* one-column Index */
											CASE
												WHEN ix.indexrelid IS NULL THEN \'\'
												ELSE
													CASE
														WHEN ix.indisprimary THEN \' PRIMARY KEY\'
														WHEN ix.indisunique  THEN \' UNIQUE\'
														ELSE \'\'
													END
											END AS "Index"
										FROM
											pg_attribute at
											INNER JOIN pg_type tp ON at.atttypid = tp.oid
											LEFT OUTER JOIN pg_attrdef ad ON ad.adrelid = tb.oid AND ad.adnum = at.attnum
											LEFT OUTER JOIN pg_index ix ON ix.indrelid = tb.oid AND ix.indnatts = 1 AND at.attnum = ix.indkey[0]
											LEFT OUTER JOIN pg_class ic ON ix.indexrelid = ic.oid
											LEFT OUTER JOIN pg_am    am ON ic.relam = am.oid
										WHERE
											tb.oid = at.attrelid AND at.attnum >= 1
									) AS columns ORDER BY attnum
								), \',\' || chr(10)
							)
							||
							(
								SELECT
									CASE
										WHEN COUNT(*) = 0 THEN \'\'
										ELSE \',\' || chr(10) || \' \' || array_to_string(
											ARRAY(
												SELECT
													CASE
														WHEN indisprimary THEN \'PRIMARY KEY \'
														ELSE \'UNIQUE \'
													END
													|| substr(indexdef, strpos(indexdef, \'(\'), strpos(indexdef, \')\') - strpos(indexdef, \'(\') + 1) || \' /* \'||index||\' */\'
												FROM
												(
													SELECT
														ic.relname AS index, ns.nspname AS schema, tb.relname AS table, ix.indnatts, ix.indisunique, ix.indisprimary, am.amname, ix.indkey, pg_get_indexdef(ic.oid) AS indexdef
													FROM
														pg_index ix
														INNER JOIN pg_class ic ON ix.indexrelid = ic.oid
														INNER JOIN pg_am    am ON ic.relam = am.oid
													WHERE
														ix.indrelid = tb.oid AND ix.indnatts > 1 AND (ix.indisprimary OR ix.indisunique)
												) AS def ORDER BY indisprimary desc, index
											), \',\'||chr(10)
										)
									END
								FROM
									pg_index ix
								WHERE
									ix.indrelid = tb.oid AND ix.indnatts > 1 AND (ix.indisprimary OR ix.indisunique)
							) || chr(10) || \');\'
						)
						END
				FROM
					pg_class tb
					INNER JOIN pg_namespace ns ON tb.relnamespace = ns.oid
				WHERE
					tb.relname = \'' . $table . '\';
			';
		} elseif ($type == 'table_define') {
			$sql = '
				SELECT
					column_name, data_type, is_nullable
				FROM
					information_schema.columns
				WHERE
					table_schema = \'public\' AND table_name = \'' . $table . '\';
			';
		}
	} elseif (DATABASE_TYPE == 'pdo_sqlite' or DATABASE_TYPE == 'pdo_sqlite2' or DATABASE_TYPE == 'sqlite') {
		if ($type == 'table_list') {
			$sql = '
				SELECT
					name
				FROM
					sqlite_master
				WHERE
					type = \'table\';
			';
		} elseif ($type == 'table_create') {
			$sql = '
				SELECT
					sql
				FROM
					sqlite_master
				WHERE
					tbl_name = \'' . $table . '\';
			';
		} elseif ($type == 'table_define') {
			$sql = '
				PRAGMA TABLE_INFO(\'' . $table . '\');
			';
		}
	}

	$sql = preg_replace('/\s+/', ' ', $sql);
	$sql = preg_replace('/^\s+/', '', $sql);
	$sql = preg_replace('/\s+$/', '', $sql);

	return $sql;
}

?>
